from django.db import models
from django.utils.text import slugify

class MainCategory(models.Model):
    name = models.CharField(max_length=100, unique=True)
    slug = models.SlugField(unique=True, blank=True)
    description = models.TextField()

    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.name)
        super().save(*args, **kwargs)

    def __str__(self):
        return self.name


class SubCategory(models.Model):
    name = models.CharField(max_length=100, unique=True)
    slug = models.SlugField(unique=True, blank=True)
    image = models.ImageField(upload_to='subcategories/', blank=True, null=True)

    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.name)
        super().save(*args, **kwargs)

    def __str__(self):
        return self.name


class Product(models.Model):
    name = models.CharField(max_length=200, unique=True)
    slug = models.SlugField(unique=True, blank=True)
    main_category = models.ForeignKey(
        MainCategory, on_delete=models.CASCADE, related_name='products'
    )  # Each product belongs to one main category
    subcategories = models.ManyToManyField(
        SubCategory, related_name='products'
    )  # Each product can belong to multiple subcategories
    image = models.ImageField(upload_to='products/', blank=True, null=True, default='default-product.png')
    description = models.TextField()
    technical_description = models.TextField(blank=True, null=True)
    type = models.CharField(max_length=100, blank=True, null=True)  # e.g., "table," "chair," etc.

    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.name)
        super().save(*args, **kwargs)

    def __str__(self):
        return self.name
